<?php
namespace Rapbit\Base\Http;


use Psr\Http\Message\MessageInterface;
use Psr\Http\Message\ResponseInterface;
use Psr\Http\Message\StreamInterface;
use Rapbit\Base\Error\BadDataException;
use Rapbit\Base\Error\DataNotFoundException;
use Rapbit\Base\Util\LogFormatter;
use Rapbit\Base\Util\StringUtil;

class Json implements ResponseInterface
{
    public function __construct(
        private int $status = 599,
        private ?array $_data = null,
        private array $headers = [],
        private string $protocolVersion = "1.1",
        private string $reasonPhrase = "",
        private string $body = ""
    ){
        $this->headers["Content-Type"] = "application/json";
        if (!is_null($this->_data)) {
            $this->body = json_encode($_data, JSON_PRETTY_PRINT);
        }
    }

    public function get(?string $arrayKey = null): mixed
    {
        if (is_null($arrayKey)){
            return $this->_data;
        }
        if (array_key_exists($arrayKey, $this->_data)){
            return $this->_data[$arrayKey];
        }
        throw new DataNotFoundException();
    }

    #[\Override] public function getStatusCode(): int
    {
        return $this->status;
    }

    #[\Override] public function getHeaders(): array
    {
        return $this->headers;
    }

    public function __toString(): string
    {
        return $this->body ?? "{}";
    }

    #[\Override] public function getProtocolVersion(): string
    {
        return $this->protocolVersion;
    }

    #[\Override] public function withProtocolVersion(string $version): MessageInterface
    {
        return new Json(
            $this->status,
            json_decode($this->body, true),
            $this->headers,
            $version
        );
    }

    #[\Override] public function hasHeader(string $name): bool
    {
        return array_key_exists($name, $this->headers);
    }

    #[\Override] public function getHeader(string $name): array
    {
        return $this->headers[$name];
    }

    #[\Override] public function getHeaderLine(string $name): string
    {
        return implode(",", $this->headers[$name]);
    }

    #[\Override] public function withHeader(string $name, $value): MessageInterface
    {
        return new Json(
            $this->status,
            json_decode($this->body, true),
            [$name => $value],
            $this->protocolVersion
        );
    }

    #[\Override] public function withAddedHeader(string $name, $value): MessageInterface
    {
        return new Json(
            $this->status,
            json_decode($this->body, true),
            array_merge([$name => $value], $this->headers),
            $this->protocolVersion
        );
    }

    #[\Override] public function withoutHeader(string $name): MessageInterface
    {
        $new = $this->headers;
        unset($new[$name]);

        return new Json(
            $this->status,
            json_decode($this->body, true),
            $new,
            $this->protocolVersion
        );
    }

    #[\Override] public function getBody(): StreamInterface
    {
        return StringUtil::stringToStream($this->body ?? "{}");
    }

    #[\Override] public function withBody(StreamInterface $body): MessageInterface
    {
        return new Json(
            $this->status,
            json_decode($body->getContents(), true),
            $this->headers,
            $this->protocolVersion
        );
    }

    #[\Override] public function withStatus(int $code, string $reasonPhrase = ''): ResponseInterface
    {
        return new Json(
            $code,
            json_decode($this->body, true),
            $this->headers,
            $this->protocolVersion,
            $reasonPhrase
        );
    }

    #[\Override] public function getReasonPhrase(): string
    {
        return $this->reasonPhrase;
    }
}