<?php
declare(strict_types=1);

namespace CPP\Base\Tests\Unit\Http;

use CPP\Base\Contract\ApiClientInterface;
use CPP\Base\Http\IdLocation;
use CPP\Base\Http\ResourceClient;
use PHPUnit\Framework\TestCase;
use Psr\Http\Message\ResponseInterface;
use Psr\Http\Message\StreamInterface;

final class ResourceClientTest extends TestCase
{
    private function jsonResponse(int $status, array $payload): ResponseInterface
    {
        $stream = new class($payload) implements StreamInterface {
            public function __construct(private array $data) {}
            public function __toString(): string { return json_encode($this->data, JSON_UNESCAPED_UNICODE); }
            public function close(): void {}
            public function detach() { return null; }
            public function getSize(): ?int { return null; }
            public function tell(): int { return 0; }
            public function eof(): bool { return true; }
            public function isSeekable(): bool { return false; }
            public function seek($offset, $whence = SEEK_SET): void {}
            public function rewind(): void {}
            public function isWritable(): bool { return false; }
            public function write($string): int { return 0; }
            public function isReadable(): bool { return true; }
            public function read($length): string { return ''; }
            public function getContents(): string { return $this->__toString(); }
            public function getMetadata($key = null): mixed { return null; }
        };

        return new class($status, $stream) implements ResponseInterface {
            public function __construct(private int $status, private StreamInterface $stream) {}
            public function getProtocolVersion(): string { return '1.1'; }
            public function withProtocolVersion($version): self { return $this; }
            public function getHeaders(): array { return []; }
            public function hasHeader($name): bool { return false; }
            public function getHeader($name): array { return []; }
            public function getHeaderLine($name): string { return ''; }
            public function withHeader($name, $value): self { return $this; }
            public function withAddedHeader($name, $value): self { return $this; }
            public function withoutHeader($name): self { return $this; }
            public function getBody(): StreamInterface { return $this->stream; }
            public function withBody(StreamInterface $body): self { return $this; }
            public function getStatusCode(): int { return $this->status; }
            public function withStatus($code, $reasonPhrase = ''): self { return $this; }
            public function getReasonPhrase(): string { return ''; }
        };
    }

    public function testQueryParamIdStyle(): void
    {
        $response = $this->jsonResponse(200, [['id' => 1]]);
        $api = new class($response) implements ApiClientInterface {
            public array $lastArgs = [];
            public function __construct(private ResponseInterface $resp) {}
            public function request(string $path, string $method = 'GET', array $query = [], array|string|null $body = null, array $headers = []): ResponseInterface
            {
                $this->lastArgs = compact('path','method','query','body','headers');
                return $this->resp;
            }
        };

        $rc = new ResourceClient($api, '/issues', IdLocation::QueryParam, 'id');
        $list = $rc->list();
        $this->assertSame('/issues', $api->lastArgs['path']);
        $this->assertSame('GET', $api->lastArgs['method']);
        $this->assertIsArray($list);

        $rc->get(5);
        $this->assertSame(['id' => 5], $api->lastArgs['query']);

        $u = $rc->update(5, ['title' => 'T']);
        $this->assertSame(['id' => 5], $api->lastArgs['query']);
        $this->assertSame(['title' => 'T'], $api->lastArgs['body']);
        $this->assertIsArray($u);
    }

    public function testPathSegmentIdStyle(): void
    {
        $response = $this->jsonResponse(200, ['ok' => true]);
        $api = new class($response) implements ApiClientInterface {
            public array $lastArgs = [];
            public function __construct(private ResponseInterface $resp) {}
            public function request(string $path, string $method = 'GET', array $query = [], array|string|null $body = null, array $headers = []): ResponseInterface
            {
                $this->lastArgs = compact('path','method','query','body','headers');
                return $this->resp;
            }
        };

        $rc = new ResourceClient($api, '/users', IdLocation::PathSegment);
        $rc->get(10);
        $this->assertSame('/users/10', $api->lastArgs['path']);
        $this->assertSame([], $api->lastArgs['query']);
    }

    public function testDeleteAccepts204(): void
    {
        $response = $this->jsonResponse(204, []);
        $api = new class($response) implements ApiClientInterface {
            public function __construct(private ResponseInterface $resp) {}
            public function request(string $path, string $method = 'GET', array $query = [], array|string|null $body = null, array $headers = []): ResponseInterface
            {
                return $this->resp;
            }
        };

        $rc = new ResourceClient($api, '/issues');
        $this->assertTrue($rc->delete(99));
    }
}