<?php
declare(strict_types=1);

namespace CPP\Base\Tests\Unit\Http;

use CPP\Base\Contract\AuthenticatorInterface;
use CPP\Base\Http\ApiClient;
use PHPUnit\Framework\TestCase;
use Psr\Http\Client\ClientInterface as HttpClientInterface;
use Psr\Http\Message\RequestFactoryInterface;
use Psr\Http\Message\StreamFactoryInterface;
use Psr\Http\Message\RequestInterface;
use Psr\Http\Message\ResponseInterface;
use Psr\Http\Message\StreamInterface;
use Rapbit\Base\Constant\HttpMethod; // Your constants [1](https://tobivisionco-my.sharepoint.com/personal/tobias_opcenter_io/Documents/Microsoft%20Copilot%20Chat%20Files/HttpMethod.php)

final class ApiClientTest extends TestCase
{
    public function testBuildsRequestWithQueryHeadersAndJsonBody(): void
    {
        $http   = $this->createMock(HttpClientInterface::class);
        $rf     = $this->createMock(RequestFactoryInterface::class);
        $sf     = $this->createMock(StreamFactoryInterface::class);
        $auth   = $this->createMock(AuthenticatorInterface::class);
        $req    = $this->createMock(RequestInterface::class);
        $resp   = $this->createMock(ResponseInterface::class);
        $stream = $this->createMock(StreamInterface::class);

        $rf->expects($this->once())
            ->method('createRequest')
            ->with(HttpMethod::POST, 'https://api.example.com/items?foo=bar&z=9')
            ->willReturn($req);

        // Default and custom headers
        $req->expects($this->exactly(2))
            ->method('withHeader')
            ->withConsecutive(
                ['Accept', 'application/json'],
                ['Content-Type', 'application/json'],
            )->willReturnSelf();

        // Authenticator gets a chance to modify the request
        $auth->expects($this->once())
            ->method('authenticate')
            ->with($req)
            ->willReturn($req);

        // JSON body becomes a PSR-7 stream
        $sf->expects($this->once())
            ->method('createStream')
            ->with('{"a":1}')
            ->willReturn($stream);

        $req->expects($this->once())
            ->method('withBody')
            ->with($stream)
            ->willReturnSelf();

        $http->expects($this->once())
            ->method('sendRequest')
            ->with($req)
            ->willReturn($resp);

        $client = new ApiClient(
            baseUrl: 'https://api.example.com',
            httpClient: $http,
            requestFactory: $rf,
            streamFactory: $sf,
            authenticator: $auth,
            defaultHeaders: ['Accept' => 'application/json']
        );

        $client->request(
            path: '/items',
            method: HttpMethod::POST,
            query: ['foo' => 'bar', 'z' => 9],
            body: ['a' => 1]
        );

        $this->assertTrue(true);
    }
}