<?php
declare(strict_types=1);

namespace CPP\Base\Tests\Unit\Auth;

use CPP\Base\Auth\ApiKeyAuthenticator;
use CPP\Base\Auth\ApiKeyLocation;
use CPP\Base\Auth\BasicAuthenticator;
use CPP\Base\Auth\BearerTokenAuthenticator;
use CPP\Base\Auth\NullAuthenticator;
use PHPUnit\Framework\TestCase;
use Psr\Http\Message\RequestInterface;
use Psr\Http\Message\UriInterface;

final class AuthenticatorsTest extends TestCase
{
    public function testBearerTokenAuthenticatorAddsAuthorizationHeader(): void
    {
        $req = $this->createMock(RequestInterface::class);
        $req->expects($this->once())
            ->method('withHeader')
            ->with('Authorization', 'Bearer TOKEN123')
            ->willReturnSelf();

        $auth = new BearerTokenAuthenticator('TOKEN123');
        $auth->authenticate($req);
        $this->assertTrue(true);
    }

    public function testBasicAuthenticatorAddsAuthorizationHeader(): void
    {
        $req = $this->createMock(RequestInterface::class);
        $req->expects($this->once())
            ->method('withHeader')
            ->with('Authorization', $this->callback(fn ($v) => str_starts_with($v, 'Basic ')))
            ->willReturnSelf();

        $auth = new BasicAuthenticator('user', 'pass');
        $auth->authenticate($req);
        $this->assertTrue(true);
    }

    public function testApiKeyHeaderAuthenticator(): void
    {
        $req = $this->createMock(RequestInterface::class);
        $req->expects($this->once())
            ->method('withHeader')
            ->with('X-API-Key', 'abc')
            ->willReturnSelf();

        $auth = new ApiKeyAuthenticator('abc', 'X-API-Key', ApiKeyLocation::Header);
        $auth->authenticate($req);
        $this->assertTrue(true);
    }

    public function testApiKeyQueryAuthenticator(): void
    {
        $uri = $this->createMock(UriInterface::class);
        $uri->method('getQuery')->willReturn('a=1');
        $uri->expects($this->once())
            ->method('withQuery')
            ->with($this->callback(fn ($q) => str_contains($q, 'a=1') && str_contains($q, 'key=abc')))
            ->willReturnSelf();

        $req = $this->createMock(RequestInterface::class);
        $req->method('getUri')->willReturn($uri);
        $req->expects($this->once())
            ->method('withUri')
            ->with($uri)
            ->willReturnSelf();

        $auth = new ApiKeyAuthenticator('abc', 'key', ApiKeyLocation::Query);
        $auth->authenticate($req);
        $this->assertTrue(true);
    }

    public function testNullAuthenticatorDoesNothing(): void
    {
        $req = $this->createMock(RequestInterface::class);
        $auth = new NullAuthenticator();
        $this->assertSame($req, $auth->authenticate($req));
    }
}