<?php
declare(strict_types=1);

namespace CPP\Base\Http;

use CPP\Base\Contract\ApiClientInterface;
use Psr\Http\Message\ResponseInterface;
use Rapbit\Base\Constant\HttpMethod;
use Rapbit\Base\Error\ApplicationError;

enum IdLocation: string { case QueryParam = 'query'; case PathSegment = 'path'; }

final class ResourceClient
{
    public function __construct(
        private readonly ApiClientInterface $api,
        private readonly string $resourcePath,
        private readonly IdLocation $idLocation = IdLocation::QueryParam,
        private readonly string $idQueryName = 'id'
    ) {}

    /** @return array<mixed> */
    public function list(array $query = []): array
    {
        $res = $this->api->request($this->resourcePath, HttpMethod::GET, $query);
        return $this->decodeJson($res, 200);
    }

    /** @return array<mixed> */
    public function get(int|string $id, array $query = []): array
    {
        [$path, $q] = $this->withId($id, $query);
        $res = $this->api->request($path, HttpMethod::GET, $q);
        return $this->decodeJson($res, 200);
    }

    /** @param array<string,mixed> $data @return array<mixed> */
    public function create(array $data): array
    {
        $res = $this->api->request($this->resourcePath, HttpMethod::POST, body: $data);
        return $this->decodeJson($res, 201, 200);
    }

    /** @param array<string,mixed> $data @return array<mixed> */
    public function update(int|string $id, array $data): array
    {
        [$path, $q] = $this->withId($id, []);
        $res = $this->api->request($path, HttpMethod::PUT, $q, $data);
        return $this->decodeJson($res, 200, 202);
    }

    public function delete(int|string $id): bool
    {
        [$path, $q] = $this->withId($id, []);
        $res = $this->api->request($path, HttpMethod::DELETE, $q);
        $code = $res->getStatusCode();
        if ($code === 204 || ($code >= 200 && $code < 300)) {
            return true;
        }
        throw new ApplicationError('Unexpected status on delete', [
            'status' => $code,
            'body'   => substr((string)$res->getBody(), 0, 500),
        ], $code);
    }

    /** @return array{0:string,1:array<string,mixed>} */
    private function withId(int|string $id, array $query): array
    {
        return match ($this->idLocation) {
            IdLocation::QueryParam => [$this->resourcePath, array_merge($query, [$this->idQueryName => $id])],
            IdLocation::PathSegment => [rtrim($this->resourcePath, '/') . '/' . rawurlencode((string)$id), $query],
        };
    }

    /** @return array<mixed> */
    private function decodeJson(ResponseInterface $res, int ...$expected): array
    {
        if ($expected && !in_array($res->getStatusCode(), $expected, true)) {
            throw new ApplicationError('Unexpected status code', [
                'expected' => $expected,
                'got' => $res->getStatusCode(),
                'body' => substr((string)$res->getBody(), 0, 500),
            ], $res->getStatusCode());
        }
        $raw = (string) $res->getBody();
        if ($raw === '') return [];
        $data = json_decode($raw, true);
        if (!is_array($data)) {
            throw new ApplicationError('Invalid JSON response: ' . substr($raw, 0, 200));
        }
        return $data;
    }
}