<?php
declare(strict_types=1);

namespace CPP\Base\Http;

use CPP\Base\Contract\ApiClientInterface;
use CPP\Base\Contract\AuthenticatorInterface;
use Psr\Http\Client\ClientInterface as HttpClientInterface;
use Psr\Http\Message\RequestFactoryInterface;
use Psr\Http\Message\StreamFactoryInterface;
use Psr\Http\Message\ResponseInterface;
use Rapbit\Base\Error\ApplicationError;
use Rapbit\Base\Constant\HttpMethod;

final class ApiClient implements ApiClientInterface
{
    /** @param array<string,string> $defaultHeaders */
    public function __construct(
        private readonly string $baseUrl,
        private readonly HttpClientInterface $httpClient,
        private readonly RequestFactoryInterface $requestFactory,
        private readonly StreamFactoryInterface $streamFactory,
        private readonly AuthenticatorInterface $authenticator,
        private readonly array $defaultHeaders = ['Accept' => 'application/json']
    ) {}

    public function request(
        string $path,
        string $method = HttpMethod::GET,
        array $query = [],
        array|string|null $body = null,
        array $headers = []
    ): ResponseInterface {
        $url = rtrim($this->baseUrl, '/') . '/' . ltrim($path, '/');

        if ($query !== []) {
            $qs = http_build_query($query);
            $url = $url . (str_contains($url, '?') ? '&' : '?') . $qs;
        }

        $request = $this->requestFactory->createRequest($method, $url);

        foreach (array_merge($this->defaultHeaders, $headers) as $name => $value) {
            $request = $request->withHeader($name, $value);
        }

        if ($body !== null) {
            if (is_array($body)) {
                $json = json_encode($body, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
                if ($json === false) {
                    throw new ApplicationError('Failed to JSON-encode request body');
                }
                if (!$request->hasHeader('Content-Type')) {
                    $request = $request->withHeader('Content-Type', 'application/json');
                }
                $stream = $this->streamFactory->createStream($json);
            } else {
                $stream = $this->streamFactory->createStream($body);
            }
            $request = $request->withBody($stream);
        }

        $request  = $this->authenticator->authenticate($request);
        $response = $this->httpClient->sendRequest($request);

        return $response;
    }
}
