<?php
declare(strict_types=1);

namespace CPP\Base\Factory;

use CPP\Base\Auth\ApiKeyAuthenticator;
use CPP\Base\Auth\ApiKeyLocation;
use CPP\Base\Auth\BasicAuthenticator;
use CPP\Base\Auth\BearerTokenAuthenticator;
use CPP\Base\Auth\NullAuthenticator;
use CPP\Base\Contract\AuthenticatorInterface;
use CPP\Base\Contract\ApiClientInterface;
use CPP\Base\Http\ApiClient;
use Psr\Http\Client\ClientInterface as HttpClientInterface;
use Psr\Http\Message\RequestFactoryInterface;
use Psr\Http\Message\StreamFactoryInterface;
use Rapbit\Base\Service\Config;
use Rapbit\Base\Error\ApplicationError;

final class ApiClientFactory
{
    public function __construct(
        private readonly Config $config,
        private readonly HttpClientInterface $httpClient,
        private readonly RequestFactoryInterface $requestFactory,
        private readonly StreamFactoryInterface $streamFactory
    ) {}

    public function create(string $baseUrlKey = 'MICROSERVICE_BASE_URL'): ApiClientInterface
    {
        $baseUrl = (string) $this->config->get($baseUrlKey, exceptionOnNotFound: true);
        $auth    = $this->buildAuthenticator();
        $headers = $this->defaultHeaders();

        return new ApiClient(
            baseUrl: $baseUrl,
            httpClient: $this->httpClient,
            requestFactory: $this->requestFactory,
            streamFactory: $this->streamFactory,
            authenticator: $auth,
            defaultHeaders: $headers
        );
    }

    /** @return array<string,string> */
    private function defaultHeaders(): array
    {
        $accept = (string) ($this->config->get('MICROSERVICE_ACCEPT', false, 'application/json'));
        $ua     = (string) ($this->config->get('MICROSERVICE_USER_AGENT', false, 'CPP-Base-Client/1.0'));
        return ['Accept' => $accept, 'User-Agent' => $ua];
    }

    private function buildAuthenticator(): AuthenticatorInterface
    {
        $bearer = (string) ($this->config->get('MICROSERVICE_BEARER_TOKEN', false, ''));
        if ($bearer !== '') return new BearerTokenAuthenticator($bearer);

        $apiKey = (string) ($this->config->get('MICROSERVICE_API_KEY', false, ''));
        if ($apiKey !== '') {
            $name = (string) ($this->config->get('MICROSERVICE_API_KEY_NAME', false, 'X-API-Key'));
            $loc  = strtolower((string) ($this->config->get('MICROSERVICE_API_KEY_IN', false, 'header')));
            $where = $loc === 'query' ? ApiKeyLocation::Query : ApiKeyLocation::Header;
            return new ApiKeyAuthenticator($apiKey, $name, $where);
        }

        $user = (string) ($this->config->get('MICROSERVICE_BASIC_USER', false, ''));
        $pass = (string) ($this->config->get('MICROSERVICE_BASIC_PASS', false, ''));
        if ($user !== '' || $pass !== '') return new BasicAuthenticator($user, $pass);

        return new NullAuthenticator();
    }

    public static function createFrom(
        Config $config,
        HttpClientInterface $httpClient,
        RequestFactoryInterface $requestFactory,
        StreamFactoryInterface $streamFactory,
        string $baseUrlKey = 'MICROSERVICE_BASE_URL'
    ): ApiClientInterface {
        $self = new self($config, $httpClient, $requestFactory, $streamFactory);
        return $self->create($baseUrlKey);
    }
}