
# CPP\\Base – Resource-Agnostic Microservice Utilities

A tiny, PSR-7/PSR-18–based helper you can inject anywhere to call any microservice. **CPP\\Base** focuses on
transport + authentication. Resource knowledge (e.g., `/issues`, `/cms/pages`) lives in the **consuming project**.

- Transport-only `ApiClient` with pluggable auth (Bearer, Basic, API key header/query, or none)
- Minimal `ResourceClient` to compose REST-y calls for a given base path and ID style (query param or path segment)
- Factory reads from your env/ini via your existing Rapbit `ConfigFactory`/`Config`
- Error handling via your `ApplicationError` with rich context

> This aligns with the style in your repository (strict types, env-driven `Config`, `HttpMethod` constants, and
> controller/tests patterns). It mirrors the endpoints shown in your Postman example and CMS controllers.

---

## Contents

- [Requirements](#requirements)
- [Installation & Autoload](#installation--autoload)
- [Configuration](#configuration)
- [Service Wiring](#service-wiring)
- [Usage](#usage)
  - [Issues microservice](#issues-microservice)
  - [CMS microservice](#cms-microservice)
  - [Path-segment vs query-parameter IDs](#path-segment-vs-query-parameter-ids)
- [Error Handling](#error-handling)
- [Testing](#testing)
  - [Unit tests](#unit-tests)
  - [Functional smoke test](#functional-smoke-test)
- [Extending](#extending)

---

## Requirements

- PHP 8.2+
- `psr/http-client`, `psr/http-message` (PSR-18 + PSR-7)
- A PSR-18 HTTP client and PSR-17 factories (e.g., Guzzle + `guzzlehttp/psr7`)
- Rapbit base services for config and errors (`Rapbit\\Base\\Service\\Config`, `Rapbit\\Base\\Error\\ApplicationError`),
  and your HTTP verb constants (`Rapbit\\Base\\Constant\\HttpMethod`).

## Installation & Autoload

Place sources under `src/Base/...` and add PSR-4 autoload in `composer.json`:

```json
{
  "autoload": {
    "psr-4": {
      "CPP\\\\Base\\\\": "src/Base/"
    }
  }
}
```

Run `composer dump-autoload`.

## Configuration

Use your existing Rapbit `ConfigFactory` to merge `.env` with an INI file and `$_ENV`.
Recognized keys (you can override the base URL key per service when building):

- `MICROSERVICE_BASE_URL` (or override with `ISSUES_BASE_URL`, `CMS_BASE_URL`, ...)
- *Auth* (choose one):
  - `MICROSERVICE_BEARER_TOKEN`
  - `MICROSERVICE_API_KEY`, `MICROSERVICE_API_KEY_NAME` (default `X-API-Key`), `MICROSERVICE_API_KEY_IN` = `header|query`
  - `MICROSERVICE_BASIC_USER`, `MICROSERVICE_BASIC_PASS`
- Optional: `MICROSERVICE_ACCEPT` (default `application/json`), `MICROSERVICE_USER_AGENT`

## Service Wiring

Example using your container style:

```php
$services['api.issues'] = static function($c) {
    /** @var \Rapbit\Base\Service\Config $cfg */
    $cfg = $c->get(\Rapbit\Base\Service\Config::class);
    return \CPP\Base\Factory\ApiClientFactory::createFrom(
        $cfg,
        $c->get(\Psr\Http\Client\ClientInterface::class),
        $c->get(\Psr\Http\Message\RequestFactoryInterface::class),
        $c->get(\Psr\Http\Message\StreamFactoryInterface::class),
        baseUrlKey: 'ISSUES_BASE_URL'
    );
};

$services['api.cms'] = static function($c) {
    /** @var \Rapbit\Base\Service\Config $cfg */
    $cfg = $c->get(\Rapbit\Base\Service\Config::class);
    return \CPP\Base\Factory\ApiClientFactory::createFrom(
        $cfg,
        $c->get(\Psr\Http\Client\ClientInterface::class),
        $c->get(\Psr\Http\Message\RequestFactoryInterface::class),
        $c->get(\Psr\Http\Message\StreamFactoryInterface::class),
        baseUrlKey: 'CMS_BASE_URL'
    );
};
```

## Usage

### Issues microservice

Create a thin wrapper in your consuming project. The Postman sample shows `/issues` supporting `GET/POST` and
`PUT/DELETE` via `?id=` query parameter.

```php
use CPP\Base\Http\ResourceClient;
use CPP\Base\Http\IdLocation;

final class IssuesClient
{
    private ResourceClient $issues;

    public function __construct(\CPP\Base\Contract\ApiClientInterface $api)
    {
        $this->issues = new ResourceClient($api, '/issues', IdLocation::QueryParam, 'id');
    }

    public function list(): array { return $this->issues->list(); }
    public function create(array $data): array { return $this->issues->create($data); }
    public function update(int $id, array $data): array { return $this->issues->update($id, $data); }
    public function delete(int $id): bool { return $this->issues->delete($id); }
}
```

### CMS microservice

Your controllers expose `/cms/pages` and `/cms/sections` with the same query-id pattern.

```php
final class CmsClient
{
    public ResourceClient $pages;
    public ResourceClient $sections;

    public function __construct(\CPP\Base\Contract\ApiClientInterface $api)
    {
        $this->pages    = new ResourceClient($api, '/cms/pages');
        $this->sections = new ResourceClient($api, '/cms/sections');
    }
}
```

### Path-segment vs query-parameter IDs

If a service uses `/resource/123` instead of `?id=123`, construct with:

```php
$users = new ResourceClient($api, '/users', IdLocation::PathSegment);
$users->get(123); // GET /users/123
```

## Error Handling

Non-expected statuses and invalid JSON throw `ApplicationError` containing HTTP status, headers, and a body snippet, aligning with your repo's error ergonomics.

## Testing

### Unit tests

- **Auth strategies** (Bearer, Basic, API key header/query, Null)
- **ApiClient** building and sending a request via PSR-18 client
- **ResourceClient** behavior for both id styles

### Functional smoke test

Optional smoke test that hits a live microservice using env variables:

- `SMOKE_BASE_URL`        – required
- `SMOKE_RESOURCE_PATH`   – e.g., `/issues` or `/cms/pages`
- `SMOKE_ID_LOCATION`     – `query` or `path` (default `query`)
- `SMOKE_ID_QUERY_NAME`   – defaults to `id`
- **Auth** via same MICROSERVICE_* env keys as the factory

The smoke test performs a `list()` call and (optionally) `create/update/delete` if you set payload envs
such as `SMOKE_CREATE_JSON`, `SMOKE_UPDATE_JSON`, and `SMOKE_DELETE_ID`.

## Extending

- Add a logging decorator to emit request/response metadata via your container logger
- Add DTO hydration if you want stronger typing in consuming projects
- Introduce retry/backoff wrappers around the PSR-18 client

---

© CPP Base utilities
